import sys, os, warnings

# This is the ovito.plugins Python package. It hosts the C++ extension modules of OVITO.

# The C++ extension modules are, however, located in a different directory when OVITO is installed
# as a conda package. For the time being, we use hardcoded, relative paths to find them.
#
# Platform-dependent paths where this Python module is located:
#   Linux:   ${PREFIX}/lib/pythonX.Y/site-packages/ovito/plugins/
#   Windows: ${PREFIX}/Lib/site-packages/ovito/plugins/
#   macOS:   ${PREFIX}/lib/pythonX.Y/site-packages/ovito/plugins/
#
# Platform-dependent paths where the native C++ shared libraries are located:
#   Linux:   ${PREFIX}/lib/ovito/plugins/
#   Windows: ${PREFIX}/Library/bin/
#   macOS:   ${PREFIX}/lib/ovito/plugins/
#

# The OVITO plugins are shared libraries, and we need to specify the path where they are found:
if sys.platform.startswith('darwin'):
    # macOS
    __path__[0] += "/../../../../ovito/plugins"
elif sys.platform.startswith('win32'):
    # Windows
    __path__[0] += "\\..\\..\\..\\..\\Library\\bin"
else:
    # Linux
    __path__[0] += "/../../../../ovito/plugins"

# On Windows, extension modules for the Python interpreter are expected to use the .pyd file extension.
# Our OVITO plugins, however, use the .ovito.dll extension. We therefore need to implement
# a custom file entry finder and hook it into the import machinery of Python.
# It specifically handles the OVITO plugin path and allows to load extension modules with .ovito.dll
# extension instead of .pyd.
if sys.platform.startswith('win32'):
    import importlib.machinery
    def OVITOPluginFinderHook(path):
        if path == __path__[0]:
            return importlib.machinery.FileFinder(path, (importlib.machinery.ExtensionFileLoader, ['.ovito.dll']))
        raise ImportError()
    sys.path_hooks.insert(0, OVITOPluginFinderHook)

# Check if an incompatible version of the Qt framework is already imported.
# If so, warn the user, because Qt5 and Qt6 cannot be used at the same time within the same application process.
_ovito_qt_version = 'Qt5'
if _ovito_qt_version == 'Qt6' and (sys.modules.get("PyQt5.QtCore") or sys.modules.get("PySide2.QtCore")):
    warnings.warn("Incompatible version of the Qt cross-platform framework detected!\nThis version of the OVITO Python module is based on Qt6 (loaded via the PySide6 bindings module), "
        "but bindings for old Qt5 are already loaded at this point (through PyQt5 or PySide2 imports preceding the import of OVITO). To avoid library version conflicts, please make sure the rest of "
        "your application uses Qt6 too instead of Qt5. "
        "\n\n"
        "To fix this warning, replace any PySide2 import statements in your script with PySide6 (or PyQt5 imports with PyQt6). "
        "In addition, it may help to set the environment variable QT_API=pyside6 to force third-party packages (e.g. matplotlib) to load Qt6 instead of Qt5. "
        "If you have any questions, please contact the OVITO developers at support@ovito.org.\n")
elif _ovito_qt_version == 'Qt5' and (sys.modules.get("PyQt6.QtCore") or sys.modules.get("PySide6.QtCore")):
    warnings.warn("Incompatible version of the Qt cross-platform framework detected! The OVITO Python module requires loading Qt5 (through the PySide2 bindings module), "
        "but some bindings for Qt6 have been loaded before (PyQt6/PySide6). To avoid ABI compatibility problems, please make sure the rest of your application "
        "also uses Qt5, not Qt6.")

# Install an import hook that will guard against incompatible Qt imports.
# Loading both Qt 5.x and 6.x into the same process leads to runtime errors.
import importlib.abc
class ImportDenier(importlib.abc.MetaPathFinder):
    def find_spec(self, fullname, path, target=None):
        if path:
            return
        if _ovito_qt_version == "Qt6" and (fullname == "PySide2" or fullname == "PyQt5"):
            raise ImportError("The ovito package you imported ealier requires PySide6 (the Python bindings for Qt6). Importing {}, which provides bindings for the incompatible Qt5 framework, "
                "is forbidden, because it leads to library version conflicts. You should update the import statements in your script to load PySide6 instead. If you have any questions, please contact support@ovito.org. ".format(fullname))
        elif _ovito_qt_version == "Qt5" and (fullname == "PySide6" or fullname == "PyQt6"):
            raise ImportError("The ovito package you imported ealier requires PySide2 (the Python bindings for Qt5). Importing {}, which provides bindings for the incompatible Qt6 framework, "
                "is forbidden, because it leads to library version conflicts. You should update the import statements in your script to load PySide2 instead. If you have any questions, please contact support@ovito.org. ".format(fullname))
sys.meta_path.insert(0, ImportDenier())

# Load Qt bindings. This is done to convince third-party packages (e.g. matplotlib), 
# which are imported by a user's script, to use the same bindings as OVITO.
from ovito.qt_compat import QtCore
from ovito.qt_compat import QtGui
