import collections.abc as collections

from . import DataCollection, DataObject, AttributeDataObject
from ..modifiers import PythonScriptModifier
from ..pipeline import StaticSource, Modifier
from ..nonpublic import PipelineStatus

# Helper class used to implement the DataCollection.attributes field.
class _AttributesView(collections.MutableMapping):

    def __init__(self, data_collection):
        """ Constructor that stores away a back-pointer to the owning DataCollection instance. """
        self._collection = data_collection

    def __len__(self):
        count = 0
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject):
                count += 1
        return count

    def __getitem__(self, key):
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject) and obj.identifier == key:
                return obj.value
        raise KeyError("Attribute '%s' does not exist in data collection." % str(key))

    def __setitem__(self, key, value):
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject) and obj.identifier == key:
                if not value is None:
                    self._collection.make_mutable(obj).value = value
                else:
                    del self._collection.objects[obj]
                return
        if not value is None:
            attr =  AttributeDataObject(identifier = key, value = value)
            self._collection.objects.append(attr)

    def __delitem__(self, key):
        """ Removes a global attribute from the data collection. """
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject) and obj.identifier == key:
                del self._collection.objects[obj]
                return
        raise KeyError("Attribute '%s' does not exist in data collection." % str(key))

    def __iter__(self):
        """ Returns an iterator over the names of all global attributes. """
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject):
                yield obj.identifier

    def __repr__(self):
        return repr(dict(self))

# Implementation of the DataCollection.attributes field.
def _DataCollection_attributes(self):
    """
    This field contains a dictionary view with all the *global attributes* currently associated with this data collection.
    Global attributes are key-value pairs that represent small tokens of information, typically simple value types such as ``int``, ``float`` or ``str``. 
    Every attribute has a unique identifier such as ``'Timestep'`` or ``'ConstructSurfaceMesh.surface_area'``. This identifier serves as look-up key in the :py:attr:`!attributes` dictionary.
    Attributes are dynamically generated by modifiers in a data pipeline or come from the data source.
    For example, if the input simulation file contains timestep information, the timestep number is made available by the :py:attr:`~ovito.pipeline.FileSource` as the
    ``'Timestep'`` attribute. It can be retrieved from pipeline's output data collection:

        >>> pipeline = import_file('snapshot_140000.dump')
        >>> pipeline.compute().attributes['Timestep']
        140000

    Some modifiers report their calculation results by adding new attributes to the data collection. See each modifier's
    reference documentation for the list of attributes it generates. For example, the number of clusters identified by the
    :py:class:`~ovito.modifiers.ClusterAnalysisModifier` is available in the pipeline output as an attribute named
    ``ClusterAnalysis.cluster_count``::

        pipeline.modifiers.append(ClusterAnalysisModifier(cutoff = 3.1))
        data = pipeline.compute()
        nclusters = data.attributes["ClusterAnalysis.cluster_count"]

    The :py:func:`ovito.io.export_file` function can be used to output dynamically computed attributes to a text file, possibly as functions of time::

        export_file(pipeline, "data.txt", "txt/attr",
            columns = ["Timestep", "ClusterAnalysis.cluster_count"],
            multiple_frames = True)

    If you are writing your own :ref:`modifier function <writing_custom_modifiers>`, you let it add new attributes to a data collection.
    In the following example, the :py:class:`~ovito.modifiers.CommonNeighborAnalysisModifier` first inserted into the
    pipeline generates the ``'CommonNeighborAnalysis.counts.FCC'`` attribute to report the number of atoms that
    have an FCC-like coordination. To compute an atomic *fraction* from that, we need to divide the count by the total number of
    atoms in the system. To this end, we append a user-defined modifier function
    to the pipeline, which computes the fraction and outputs the value as a new attribute named ``'fcc_fraction'``.

    .. literalinclude:: ../example_snippets/python_modifier_generate_attribute.py
        :lines: 6-

    """
    return _AttributesView(self)
DataCollection.attributes = property(_DataCollection_attributes)

# Implementation of the DataCollection.apply() method:
def __DataCollection_apply(self, modifier, frame = None):
    """ This method applies a :py:class:`~ovito.pipeline.Modifier` function to the data in this collection in place.

        It allows modifying a data collection with one of Ovito's built-in modifiers directly without the need to build up a complete
        :py:class:`~ovito.pipeline.Pipeline` first. In contrast to a data pipeline, the :py:meth:`!apply()` method
        executes the modifier function immediately and changes the input data in place. In other words, the original data in this :py:class:`DataCollection`
        will be replaced by the output produced by the invoked modifier function. Note that it is possible to create a copy of
        the original data collection using the :py:meth:`.clone` method if needed. The following code example
        demonstrates how to use the :py:meth:`!apply()` method to successively modify the state of a dataset:

        .. literalinclude:: ../example_snippets/data_collection_apply.py
            :lines: 4-10

        Note that it is typically possible to achieve the same result by first populating a :py:class:`~ovito.pipeline.Pipeline` with the modifiers and then calling its
        :py:meth:`~ovito.pipeline.Pipeline.compute` method at the very end:

        .. literalinclude:: ../example_snippets/data_collection_apply.py
            :lines: 15-19

        Also note that :py:meth:`!apply()` may be called from a user-defined modifier function (see :py:class:`~ovito.modifiers.PythonScriptModifier`)
        in order to invoke a built-in modifier as a sub-routine:

        .. literalinclude:: ../example_snippets/data_collection_apply.py
            :lines: 24-34

        :param modifier: The :py:class:`~ovito.pipeline.Modifier` that will be called by the method to alter the data in place.
        :param int frame: An optional animation frame number that will be passed to the modifier function, which may use it to implement time-dependent modifications.
    """

    # This method expects the Modifier to be applied.
    if not isinstance(modifier, Modifier):
        if callable(modifier):
            # Automatically wrap Python methods in a PythonScriptModifier object.
            modifier = PythonScriptModifier(function = modifier)
        else:
            raise TypeError("Expected a modifier as argument")

    # Build an ad-hoc pipeline by creating a ModifierApplication for the Modifier, which is
    # connected to a StaticSource delivering this input DataCollection.
    modapp = modifier.create_modifier_application()
    modapp.input = StaticSource(data = self)
    modapp.modifier = modifier

    # Determine the animation time at which to evaluate the modifier.
    if frame is not None:
        time = modapp.source_frame_to_anim_time(frame)
    else:
        time = modifier.dataset.anim.time

    # initialize the modifier in the pipeline.
    modifier.initialize_modifier(time, modapp)

    # Evaluate the ad-hoc pipeline.
    state = modapp._evaluate(time)
    if state.status.type == PipelineStatus.Type.Error:
        raise RuntimeError("Modifier evaluation failed: %s" % state.status.text)

    # The DataCollection.apply() method is supposed to modify the DataCollection in place.
    # To implement this behavior, move the data objects from the pipeline output collection
    # over to this collection, replacing the original state.
    self._assign_objects(state.data)
DataCollection.apply = __DataCollection_apply

# Implementation of the DataCollection.clone() method:
def __DataCollection_clone(self):
    """
        Returns a copy of this :py:class:`DataCollection` containing the same data objects as the original.

        The method may be used to retain a copy of the original data before modifying a data collection in place,
        for example using the :py:meth:`.apply` method:

        .. literalinclude:: ../example_snippets/data_collection_clone.py
            :lines: 8-12

        Note that the :py:meth:`!clone()` method performs an inexpensive, shallow copy, meaning that the newly created collection will still share
        the data objects with the original collection. Data objects that are shared by two or more data collections are
        protected against modification by default to avoid unwanted side effects.
        Thus, in order to subsequently modify the data objects in either the original collection or its
        copy, you will have to use the underscore notation or the :py:meth:`DataObject.make_mutable` method to explicitly
        request a deep copy of the particular data object(s) you want to modify. For example:

        .. literalinclude:: ../example_snippets/data_collection_clone.py
            :lines: 17-28
    """
    cloned_collection = DataCollection()
    cloned_collection._assign_objects(self)
    return cloned_collection
DataCollection.clone = __DataCollection_clone

# Implementation of the DataCollection._find_object_type() method.
# Internal method used for looking up a certain type of DataObject in a DataCollection.
def _DataCollection_find_object_type(self, data_obj_class):
    assert(issubclass(data_obj_class, DataObject))
    for obj in self.objects:
        if isinstance(obj, data_obj_class):
            return obj
    return None
DataCollection._find_object_type = _DataCollection_find_object_type
