import collections.abc as collections

from . import DataCollection, DataObject, AttributeDataObject, TriangleMesh
from ._data_objects_dict import DataObjectsDict
from ..modifiers import PythonModifier
from ..pipeline import StaticSource, Modifier
from ..nonpublic import PipelineStatus
import ovito

# Helper class used to implement the DataCollection.attributes field.
class _AttributesView(collections.MutableMapping):

    def __init__(self, data_collection):
        """ Constructor that stores away a back-pointer to the owning DataCollection instance. """
        self._collection = data_collection

    def __len__(self):
        count = 0
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject):
                count += 1
        return count

    def __getitem__(self, key):
        if not isinstance(key, str):
            raise TypeError("Attribute key must be a string")
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject) and obj.identifier == key:
                return obj.value
        raise KeyError(f"Attribute '{key}' does not exist in data collection.")

    def __setitem__(self, key, value):
        if not isinstance(key, str):
            raise TypeError("Attribute key must be a string")
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject) and obj.identifier == key:
                if not value is None:
                    self._collection.make_mutable(obj).value = value
                else:
                    self._collection.objects.remove(obj)
                return
        if not value is None:
            self._collection.objects.append(AttributeDataObject(identifier = key, value = value))

    def __delitem__(self, key):
        """ Removes a global attribute from the data collection. """
        if not isinstance(key, str):
            raise TypeError("Attribute key must be a string")
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject) and obj.identifier == key:
                self._collection.objects.remove(obj)
                return
        raise KeyError(f"Attribute '{key}' does not exist in data collection.")

    def __iter__(self):
        """ Returns an iterator over the names of all global attributes. """
        for obj in self._collection.objects:
            if isinstance(obj, AttributeDataObject):
                yield obj.identifier

    def __repr__(self):
        return repr(dict(self))

# Implementation of the DataCollection.attributes field.
def _DataCollection_attributes(self):
    """
    This field contains a dictionary view with all the *global attributes* currently associated with this data collection.
    Global attributes are key-value pairs that represent small tokens of information, typically simple value types such as ``int``, ``float`` or ``str``.
    Every attribute has a unique identifier such as ``'Timestep'`` or ``'ConstructSurfaceMesh.surface_area'``. This identifier serves as lookup key in the :py:attr:`!attributes` dictionary.
    Attributes are dynamically generated by modifiers in a data pipeline or come from the data source.
    For example, if the input simulation file contains timestep information, the timestep number is made available by the :py:attr:`~ovito.pipeline.FileSource` as the
    ``'Timestep'`` attribute. It can be retrieved from pipeline's output data collection:

        >>> pipeline = import_file('snapshot_140000.dump')
        >>> pipeline.compute().attributes['Timestep']
        140000

    Some modifiers report their calculation results by adding new attributes to the data collection. See each modifier's
    reference documentation for the list of attributes it generates. For example, the number of clusters identified by the
    :py:class:`~ovito.modifiers.ClusterAnalysisModifier` is available in the pipeline output as an attribute named
    ``ClusterAnalysis.cluster_count``::

        pipeline.modifiers.append(ClusterAnalysisModifier(cutoff = 3.1))
        data = pipeline.compute()
        nclusters = data.attributes["ClusterAnalysis.cluster_count"]

    The :py:func:`ovito.io.export_file` function can be used to output dynamically computed attributes to a text file, possibly as functions of time::

        export_file(pipeline, "data.txt", "txt/attr",
            columns = ["Timestep", "ClusterAnalysis.cluster_count"],
            multiple_frames = True)

    If you are writing your own :ref:`modifier function <writing_custom_modifiers>`, you let it add new attributes to a data collection.
    In the following example, the :py:class:`~ovito.modifiers.CommonNeighborAnalysisModifier` first inserted into the
    pipeline generates the ``'CommonNeighborAnalysis.counts.FCC'`` attribute to report the number of atoms that
    have an FCC-like coordination. To compute an atomic *fraction* from that, we need to divide the count by the total number of
    atoms in the system. To this end, we append a user-defined modifier function
    to the pipeline, which computes the fraction and outputs the value as a new attribute named ``'fcc_fraction'``.

    .. literalinclude:: ../example_snippets/python_modifier_generate_attribute.py
        :lines: 6-

    """
    return _AttributesView(self)
DataCollection.attributes = property(_DataCollection_attributes)

# Implementation of the DataCollection.triangle_meshes attribute.
def _DataCollection_triangle_meshes(self):
    """
    This is a dictionary view providing key-based access to all :py:class:`TriangleMesh` objects currently stored in
    this data collection. Each :py:class:`TriangleMesh` has a unique :py:attr:`~ovito.data.DataObject.identifier` key,
    which can be used to look it up in the dictionary.
    """
    return DataObjectsDict(self, TriangleMesh)
DataCollection.triangle_meshes = property(_DataCollection_triangle_meshes)

# Implementation of the DataCollection.apply() method:
def __DataCollection_apply(self, modifier, frame = None):
    """ This method applies a :py:class:`~ovito.pipeline.Modifier` function to the data stored in this collection to modify it in place.

        :param ovito.pipeline.Modifier modifier: The modifier object that should alter the contents of this data collection in place.
        :param int frame: Optional animation frame number to be passed to the modifier function, which may use it for time-dependent modifications.

        The method allows modifying a data collection with one of OVITO's modifiers directly without the need to build up a complete
        :py:class:`~ovito.pipeline.Pipeline` first. In contrast to a :ref:`data pipeline <modifiers_overview>`, the :py:meth:`!apply()` method
        executes the modifier function immediately and alters the data in place. In other words, the original data in this :py:class:`DataCollection`
        gets replaced by the output produced by the invoked modifier function. It is possible to first create a copy of
        the original data using the :py:meth:`.clone` method if needed. The following code example
        demonstrates how to use :py:meth:`!apply()` to successively modify a dataset:

        .. literalinclude:: ../example_snippets/data_collection_apply.py
            :lines: 4-10

        Note that it is typically possible to achieve the same result by first populating a :py:class:`~ovito.pipeline.Pipeline` with the modifiers and then calling its
        :py:meth:`~ovito.pipeline.Pipeline.compute` method at the very end:

        .. literalinclude:: ../example_snippets/data_collection_apply.py
            :lines: 15-19

        An important use case of the :py:meth:`!apply()` method is in the implementation of a :ref:`user-defined modifier function <writing_custom_modifiers>`,
        making it possible to invoke other modifiers as sub-routines:

        .. literalinclude:: ../example_snippets/data_collection_apply.py
            :lines: 24-34
    """

    # This method expects a Modifier object as argument.
    if not isinstance(modifier, Modifier):
        if isinstance(modifier, ovito.pipeline.ModifierInterface):
            # Automatically wrap ModifierInterface objects in a PythonModifier object.
            modifier = PythonModifier(delegate=modifier)
        elif callable(modifier):
            # Automatically wrap freestanding Python methods in a PythonModifier object.
            modifier = PythonModifier(function=modifier)
        else:
            raise TypeError("Expected a modifier as argument")

    # Build an ad-hoc pipeline by creating a ModificationNode for the Modifier,
    # which receives the input DataCollection from a StaticSource.
    node = modifier.create_modification_node()
    node.input = StaticSource(data=self)
    node.modifier = modifier

    # Initialize the modifier within the pipeline, e.g. parameters that depend on the modifier's input.
    modifier.initialize_modifier(node, frame)

    # Evaluate the ad-hoc pipeline.
    state = node._evaluate(frame)
    if state.status.type == PipelineStatus.Type.Error:
        raise RuntimeError(f"Modifier evaluation failed: {state.status.text}")

    # The DataCollection.apply() method is supposed to modify the DataCollection in place.
    # To implement this behavior, move the data objects from the pipeline output collection
    # over to this collection, replacing the original state.
    self._assign_objects(state.data)
DataCollection.apply = __DataCollection_apply

# Implementation of the DataCollection.clone() method:
def __DataCollection_clone(self):
    """
        Returns a copy of this :py:class:`DataCollection` containing the same data objects as the original.

        The method may be used to retain a copy of the original data before modifying a data collection in place,
        for example using the :py:meth:`.apply` method:

        .. literalinclude:: ../example_snippets/data_collection_clone.py
            :lines: 8-12

        Note that the :py:meth:`!clone()` method performs an inexpensive, shallow copy, meaning that the newly created collection will still share
        the data objects with the original collection. Data objects that are shared by two or more data collections are
        protected against modification by default to avoid unwanted side effects.
        Thus, in order to subsequently modify the data objects in either the original collection or its
        copy, you will have to use the underscore notation or the :py:meth:`DataObject.make_mutable` method to explicitly
        request a deep copy of the particular data object(s) you want to modify. For example:

        .. literalinclude:: ../example_snippets/data_collection_clone.py
            :lines: 17-28
    """
    cloned_collection = DataCollection()
    cloned_collection._assign_objects(self)
    return cloned_collection
DataCollection.clone = __DataCollection_clone

# Implementation of the DataCollection._find_object_type() method.
# Internal method used for looking up a certain type of DataObject in a DataCollection.
def _DataCollection_find_object_type(self, data_obj_class):
    assert(issubclass(data_obj_class, DataObject))
    for obj in self.objects:
        if isinstance(obj, data_obj_class):
            return obj
    return None
DataCollection._find_object_type = _DataCollection_find_object_type
