import sys
import warnings

# This is the ovito.plugins Python package. It hosts the C++ extension modules of OVITO.

# The C++ extension modules are, however, located in a different directory when OVITO is installed
# as a conda package. For the time being, we use hardcoded, relative paths to find them.
#
# Platform-dependent paths where this Python module is located:
#   Linux:   ${PREFIX}/lib/pythonX.Y/site-packages/ovito/plugins/
#   Windows: ${PREFIX}/Lib/site-packages/ovito/plugins/
#   macOS:   ${PREFIX}/lib/pythonX.Y/site-packages/ovito/plugins/
#
# Platform-dependent paths where the native C++ shared libraries are located:
#   Linux:   ${PREFIX}/lib/ovito/plugins/
#   Windows: ${PREFIX}/Library/bin/
#   macOS:   ${PREFIX}/lib/ovito/plugins/
#

# The OVITO plugins are shared libraries, and we need to specify the path where they are found:
if 'ON' == 'OFF' or not getattr(sys, "_ovito_embedded_mode", False):
    if sys.platform.startswith('darwin'):
        # macOS
        __path__[0] += "/../../../../ovito/plugins"
    elif sys.platform.startswith('win32'):
        # Windows
        __path__[0] += "\\..\\..\\..\\..\\Library\\bin"
    else:
        # Linux
        __path__[0] += "/../../../../ovito/plugins"

# On Windows, extension modules for the Python interpreter are expected to use the .pyd file extension.
# Our OVITO plugins, however, use the .ovito.dll extension. We therefore need to implement
# a custom file entry finder and hook it into the import machinery of Python.
# It specifically handles the OVITO plugin path and allows to load extension modules with .ovito.dll
# extension instead of .pyd.
if 'ON' == 'OFF' and sys.platform.startswith('win32'):
    import importlib.machinery
    def OVITOPluginFinderHook(path):
        if path == __path__[0]:
            return importlib.machinery.FileFinder(path, (importlib.machinery.ExtensionFileLoader, ['.ovito.dll']))
        raise ImportError()
    sys.path_hooks.insert(0, OVITOPluginFinderHook)

# If all plugins were statically linked into the main OVITO executable,
# make them loadable as built-in modules below the ovito.plugins package.
if 'ON' == 'ON' and getattr(sys, "_ovito_embedded_mode", False):
    import importlib.machinery
    def OVITOBuiltinPluginFinderHook(path):
        if path == __path__[-1]:
            return importlib.machinery.BuiltinImporter()
        raise ImportError()
    sys.path_hooks.insert(0, OVITOBuiltinPluginFinderHook)

# Check if an incompatible version of the Qt framework is already imported.
# If so, warn the user, because Qt5 and Qt6 cannot be used at the same time within the same application process.
if sys.modules.get("PyQt5.QtCore") or sys.modules.get("PySide2.QtCore"):
    warnings.warn("Incompatible version of the Qt cross-platform framework detected!\nThis version of the OVITO Python module is based on Qt6 (loaded via the PySide6 bindings module), "
        "but bindings for old Qt5 are already loaded at this point (through PyQt5 or PySide2 imports preceding the import of OVITO). To avoid library version conflicts, please make sure the rest of "
        "your application uses Qt6 too instead of Qt5. "
        "\n\n"
        "To fix this warning, replace any PySide2 import statements in your script with PySide6 (or PyQt5 imports with PyQt6). "
        "In addition, it may help to set the environment variable QT_API=pyside6 to force third-party packages (e.g. matplotlib) to load Qt6 instead of Qt5. "
        "If you have any questions, please contact the OVITO developers at support@ovito.org.\n")

# Install an import hook that will guard against incompatible Qt imports.
# Loading both Qt5 and Qt6 into the same process leads to runtime conflicts.
import importlib.abc
class ImportDenier(importlib.abc.MetaPathFinder):
    def find_spec(self, fullname, path, target=None):
        if path:
            return
        if fullname in ("PySide2", "PyQt5"):
            raise ImportError(f"The ovito package you imported earlier requires PySide6 (the Python bindings for Qt6). Importing {fullname}, which provides bindings for the incompatible Qt5 framework, "
                "is forbidden, because it leads to library version conflicts. You should update the import statements in your script to load PySide6 instead. If you have any questions, please contact support@ovito.org.")
sys.meta_path.insert(0, ImportDenier())

# When we are running in an external Python interpreter, load the monolithic C++ extension
# module containing the OVITO bindings.
if 'ON' == 'ON' and not getattr(sys, "_ovito_embedded_mode", False):
    import ovito.plugins.ovito_bindings

# Load Qt bindings. This is done to convince third-party packages (e.g. matplotlib),
# which are imported by a user's script, to use the same bindings as OVITO.
from ovito.qt_compat import QtCore
from ovito.qt_compat import QtGui
